from os import listdir, makedirs, system, rename, remove
from os.path import isfile, join, exists
import time
import logging
import threading
import json
from datetime import datetime
from sherlock import Timer
from urServiceFileGenerator import UrServiceFileGenerator

global logger
logger = logging.getLogger('daemon.' + __name__)

class WearAnalysisRecorderService :
    def __init__(self):
        self.recv_thread = None
        if exists("/root/ur-serial") is True:
            self.sherlock_dir = '/data/myUR/.sherlock/'
            self.urservice_dir = '/data/myUR/ur_service/'
            self.root_dir = "/root"
            self.isURSim = False
        else:
            self.sherlock_dir = '/tmp/myUR/.sherlock/'
            self.urservice_dir = '/tmp/myUR/ur_service/'
            self.root_dir = "/ursim"
            self.isURSim = True

        self.serial = self._get_serial()

        self._create_dir(self.sherlock_dir)

        self.file_generator = UrServiceFileGenerator()

        self.upload_to_blob_service = None
        self.ur_service_file_prefix = "ur_service_"
        self.state_file_name_full_path = join(self.sherlock_dir, "sherlock_state.json")
        self.state = self._get_state_from_file()
        self.__init_timer()

    def _get_default_configuration(self):
        state = {}
        state["enabled"] = True
        state["schedule"] = {
            "weeks": 1,
            "days": 0,
            "hours": 0,
            "minutes": 0
        }
        state["recording_length"] = 90
        return state

    def _create_config_file_if_not_exists(self):
        if not exists(self.state_file_name_full_path):
            state = self._get_default_configuration()
            self._write_to_state_file(state)

    def _write_to_state_file(self, state):
        with open(self.state_file_name_full_path, 'w') as f:
            f.write(json.dumps(state))

    def _get_state_from_file(self):
        self._create_config_file_if_not_exists()

        with open(self.state_file_name_full_path, "r") as f:
            state = json.loads(f.read())
            if "schedule" not in state:
                default_state = self._get_default_configuration()
                state["schedule"] = default_state["schedule"]
                self._write_to_state_file(state)
            if "enabled" not in state:
                default_state = self._get_default_configuration()
                state["enabled"] = default_state["enabled"]
                self._write_to_state_file(state)
            if "recording_length" not in state:
                default_state = self._get_default_configuration()
                state["recording_length"] = default_state["recording_length"]
                self._write_to_state_file(state)
        return state

    def __init_timer(self):
        schedule = self.state["schedule"]
        self.timer1 = Timer(self.sherlock_dir,
                            schedule["weeks"],
                            schedule["days"],
                            schedule["hours"],
                            schedule["minutes"])

    def start(self):
        self.recv_thread = threading.Thread(target=self._run)
        self.recv_thread.daemon = True
        self.recv_thread.start()

    def _get_serial(self):
        serial = ''
        serial_files = ['/ursim/programs/usbdisk/myUR-serial',
                        '/programs/usbdisk/myUR-serial',
                        '/root/ur-serial',
                        '/home/ur/ursim-current/ur-serial']

        for file in serial_files:
            if exists(file):
                with open(file, 'r') as fp:
                    serial = fp.read().strip('\n')
                    return serial
        return serial

    def _create_dir(self, dir):
        if not exists(dir):
            makedirs(dir)

    def _generate_and_upload(self):
        wait_for_movement = True
        try:
            self._cleanup_previous_file()
            length = self.state["recording_length"]
            filename = self.ur_service_file_prefix + self.serial + '_' + datetime.now().strftime('%Y-%m-%d_%H-%M-%S')
            file = self.file_generator.generate(self.sherlock_dir, filename, wait_for_movement, length)
            self._uploadFileToBlob(file)
        except Exception as e:
            logger.error("Failed to generate ur_service file: " + str(e))
            return False
        return True

    def generate_and_upload_manual(self):
        self._create_dir(self.urservice_dir)
        file = self.__generate_file(self.urservice_dir)
        self._uploadFileToBlob(file, "UrServiceFiles", 'Manual')
        return True

    def generate_for_usb_copy(self, filename):
        self._create_dir(self.urservice_dir)
        self.__generate_file(self.urservice_dir, filename)
        return True

    def __generate_file(self, file_directory_path, filename=""):
        wait_for_movement = False
        self._cleanup_previous_file()
        length = self.state["recording_length"]
        if not bool(filename):
            filename = self.ur_service_file_prefix + self.serial + '_' + datetime.now().strftime('%Y-%m-%d_%H-%M-%S')
        file = self.file_generator.generate(file_directory_path, filename, wait_for_movement, length)
        return file

    def _run(self):
        time.sleep(10)
        while True:
            if self.is_monitoring_enabled() and self.timer1.non_blocking_wait():
                success = self._generate_and_upload()
                if success:
                    self.timer1.reset()

            time.sleep(60)

    def is_monitoring_enabled(self):
        return self.state["enabled"]

    def get_status(self):
        return self.state

    def _convert_int(self, value):
        if value is None:
            return 0
        try:
            return int(value)
        except ValueError:
            return 0

    def set_recording_length(self, length):
        value = self._convert_int(length)
        if value == 0:
            raise Exception("Invalid recording length: " + str(length))
        self.state["recording_length"] = value
        self._write_to_state_file(self.state)
        logger.info("Recording length changed to " + str(value) + " seconds")
        return True

    def set_recording_interval(self, weeks, days, hours, minutes):
        weeks = self._convert_int(weeks)
        days = self._convert_int(days)
        hours = self._convert_int(hours)
        minutes = self._convert_int(minutes)
        if weeks==0 and days==0 and hours==0 and minutes==0:
            weeks = 1

        self.timer1.update_schedule(weeks, days, hours, minutes)
        self.state["schedule"] = {
            "weeks": weeks,
            "days": days,
            "hours": hours,
            "minutes": minutes
        }
        self._write_to_state_file(self.state)
        logger.info("New recording schedule every " + str(weeks) + " weeks, " + str(days) + " days, " + str(hours) +
                     " hours, " + str(minutes) + " minutes")


    def enable_monitoring(self, enable):
        self.state["enabled"] = enable
        self._write_to_state_file(self.state)
        result = self.is_monitoring_enabled()
        if result == True:
            logger.info("Monitoring enabled")
        else:
            logger.info("Monitoring disabled")
        return result

    def set_connection_details(self, upload_to_blob_handler):
        self.upload_to_blob_service = upload_to_blob_handler

    def on_connection_state_change(self):
        #logger.debug("Connected to IoT Hub, checking for pending sherlock files")
        self._bunch_upload_backups_when_connected()

    def _cleanup_previous_file(self):
        all_files = listdir(self.sherlock_dir)
        for file in all_files:
            file_abs_path = join(self.sherlock_dir, file)
            if isfile(file_abs_path) and file.startswith(self.ur_service_file_prefix):
                remove(file_abs_path)

    def _bunch_upload_backups_when_connected(self):
        all_files = listdir(self.sherlock_dir)
        for file in all_files:
            file_abs_path = join(self.sherlock_dir, file)
            if isfile(file_abs_path) and file.startswith(self.ur_service_file_prefix):
                self._uploadFileToBlob(file_abs_path)

    def _uploadFileToBlob(self, abs_file_path, blob_folder="Sherlock", service_file_trigger='Automatic'):
        was_upload_successful = False
        if (self.upload_to_blob_service is not None and self.upload_to_blob_service.is_device_connected() is True) :
            logger.debug("Uploading Sherlock file to blob storage: " + abs_file_path)
            tags = {'SerialNumber': self.serial, 'Name': 'UrServiceFile', 'UrServiceFileTrigger': service_file_trigger}
            was_upload_successful = self.upload_to_blob_service.upload_files(blob_folder, abs_file_path, tags, True)
            if was_upload_successful is True :
                logger.debug("Removing file after successful upload: " + abs_file_path)
            else:
                logger.warn("_uploadFileToBlob failed for " + abs_file_path)
            remove(abs_file_path)
        return was_upload_successful
