import logging
import imp
import json
import urllib2 as urlreq
from iotHubConnectionHandler import IotHubConnectionHandler
from autoConnectService import AutoConnectService

logger = logging.getLogger('daemon.' + __name__)

class URCapMediator:

    def __init__(self):
        # Direct logging output to stdout. Without adding a handler, no logging output is visible
        logger.info("URCapMediator initialized")
        self.iotHubConnectionHandler = IotHubConnectionHandler()
        self.isDeviceClientInitialized = False
        self.telemetry = None
        self.message_handler = None
        self.remote_access_handler = None
        self.remote_update_handler = None
        self.urcap_update_handler = None
        self.upload_to_blob_handler = None
        self.program_directory_backup_service = None
        self.wear_analysis_recorder_service = None
        self.check_for_update_url = ""
        self.start_auto_connect_service()
        self.start_backup_service()
        self.start_wear_analysis_recorder_service()

    def connect_device_client(self, product_info=""):
        if not self.isDeviceClientInitialized:
            self.isDeviceClientInitialized = self.iotHubConnectionHandler.init_device_client(product_info)
            if self.isDeviceClientInitialized:
                from iotHubMessageHandler import IotHubMessageHandler
                from telemetryHandler import TelemetryHandler
                from remoteAccessHandler import RemoteAccessHandler
                from urcapUpdateHandler import URCapUpdateHandler
                from uploadToBlobHandler import UploadToBlobHandler
                from remoteUpdateHandler import RemoteUpdateHandler
                self.telemetry = TelemetryHandler(self.iotHubConnectionHandler.device_client)
                self.upload_to_blob_handler = UploadToBlobHandler(self.iotHubConnectionHandler.device_client)
                self.remote_access_handler = RemoteAccessHandler(self.iotHubConnectionHandler.device_client)
                self.remote_update_handler = RemoteUpdateHandler(self.iotHubConnectionHandler.device_client)
                self.urcap_update_handler = URCapUpdateHandler(self.iotHubConnectionHandler.device_client)
                self.message_handler = IotHubMessageHandler(self.iotHubConnectionHandler.device_client, self.upload_to_blob_handler, self.remote_update_handler, self.wear_analysis_recorder_service)
            else:
                return False

        self.iotHubConnectionHandler.connect_device_client()
        return True

    def disconnect_device_client(self):
        self.iotHubConnectionHandler.disconnect_device_client()
        return True

    def init_listeners(self, check_for_update_url):
        self.check_for_update_url = check_for_update_url
        self.start_backup_service()
        if self.program_directory_backup_service is not None:
            self.program_directory_backup_service.set_connection_details(self.upload_to_blob_handler)
        if self.wear_analysis_recorder_service is not None:
            self.wear_analysis_recorder_service.set_connection_details(self.upload_to_blob_handler)
        self.message_handler.subscribe_listening_for_messages()
        self.iotHubConnectionHandler.device_client.on_connection_state_change = self._subscribe_to_connection_state_changes
        return True

    def _subscribe_to_connection_state_changes(self):
        if self.iotHubConnectionHandler.device_client.connected is True:
            if self.program_directory_backup_service is not None:
                self.program_directory_backup_service.on_connection_state_change()
            if self.wear_analysis_recorder_service is not None:
                self.wear_analysis_recorder_service.on_connection_state_change()
            self.urcap_update_handler.connection_state_changed(self.check_for_update_url)

    def get_connection_string(self, url):
        return self.iotHubConnectionHandler.get_connection_string(url)

    def send_message(self, message):
        return self.telemetry.send_message(message)

    def send_allow_access_response(self, correlationId, accepted):
        self.remote_access_handler.send_remote_request_access_response(correlationId, self._parse_bool_string(accepted))
        return True

    def send_remote_access_client_connected_event(self, correlationId):
        self.remote_access_handler.send_remote_access_client_connected_event(correlationId)
        return True

    def close_remote_access_session(self, correlationId):
        self.remote_access_handler.close_ssh_tunnel(correlationId)
        return True

    def create_tunnel(self, correlationId, domain):
        try :
            return self.remote_access_handler.create_ssh_tunnel(str(correlationId), str(domain))
        except Exception as e:
            logger.error("Error in creating tunnel: " + str(e))
            return False

    def start_software_update(self, software_version):
        self.remote_update_handler.try_start_update(str(software_version))
        return True

    def _parse_bool_string(self, bool_response):
        value = False
        if str(bool_response).lower() == "true" :
            value = True
        return value

    def start_auto_connect_service(self):
        try:
            self.auto_connect_service = AutoConnectService(self.iotHubConnectionHandler)
            logger.debug ("Done initiating auto connect service")
        except Exception:
            logger.exception("Error starting auto connect service")
            self.auto_connect_service = None

    def start_backup_service(self):
        if self.program_directory_backup_service is None:
            try:
                import watchdog
                from programDirectoryBackupService import ProgramDirectoryFilesBackupService
                self.program_directory_backup_service = ProgramDirectoryFilesBackupService()
                logger.debug ("Done initiating backup service")
            except Exception as e:
                self.program_directory_backup_service = None
                imp.reload(watchdog)

    def start_wear_analysis_recorder_service(self):
        logger.info("start_wear_analysis_recorder_service")
        if self.wear_analysis_recorder_service is None:
            try:
                from wearAnalysisRecorderService import WearAnalysisRecorderService
                self.wear_analysis_recorder_service = WearAnalysisRecorderService()
                self.wear_analysis_recorder_service.start()
                logger.debug ("Done initiating wear analysis recorder service")
            except Exception as e:
                logger.error("Error starting wear analysis recorder service: " + str(e))
                self.wear_analysis_recorder_service = None


    def upload_urcap_logs(self, api_url, abs_path):
        blob_storage_folder = "MyURCapLogs"
        if self.upload_to_blob_handler is None or not self.upload_to_blob_handler.is_device_connected():
            return self.__upload_files_while_disconnected(abs_path, api_url)
        return self.upload_to_blob_handler.upload_any_file(blob_storage_folder, abs_path, api_url, True)

    def upload_service_file(self):
        if self.wear_analysis_recorder_service is not None:
            self.wear_analysis_recorder_service.generate_and_upload_manual()
        return True

    def generate_service_file_for_usb(self, filename):
        if self.wear_analysis_recorder_service is not None:
            self.wear_analysis_recorder_service.generate_for_usb_copy(filename)
        return True

    def __upload_files_while_disconnected(self, abs_path, api_url):
        from azure.storage.blob import BlobClient
        sas_url = self._get_blob_info(api_url)
        with BlobClient.from_blob_url(sas_url) as blob_client:
            with open(str(abs_path), "rb") as f:
                result = blob_client.upload_blob(f, overwrite=True)
                logger.debug("Upload succeeded. Result is: " + str(result))
            return True
        return False

    def _get_blob_info(self, api_url):
        try:
            req = urlreq.Request(api_url)
            response = urlreq.urlopen(req).read()
            upload_info_json = json.loads(response)
            return upload_info_json['blobUrl']
        except Exception as err:
            logger.error("Error getting blob info to upload logs: " + str(err))
