import glob
import os
import json
from azure.storage.blob import BlobClient
import ast
import logging

class UploadToBlobHandler:

    def __init__(self, device_client):
        global logger
        logger = logging.getLogger('daemon.' + __name__)
        self.device_client = device_client

    def _store_blob(self, storage_info, file_name, tags, should_attach_metadata):
        try:
            # Upload the specified file
            sas_url = "https://{}/{}/{}{}".format(
                storage_info["hostName"],
                storage_info["containerName"],
                storage_info["blobName"],
                storage_info["sasToken"],
            )

            logger.info(
                "\nUploading file: {} to Azure Storage as blob: {} in container {}\n".format(
                    file_name, storage_info["blobName"], storage_info["containerName"]
                )
            )

            with BlobClient.from_blob_url(sas_url) as blob_client:
                with open(str(file_name), "rb") as f:
                    result = blob_client.upload_blob(f, overwrite=True)
                    if should_attach_metadata:
                        try:
                            if type(tags) is dict:
                                blob_client.set_blob_metadata(tags)
                            else:
                                tags = ast.literal_eval(str(tags))
                                blob_client.set_blob_metadata(tags)
                        except:
                            try:
                                tags = json.loads(tags)
                                blob_client.set_blob_metadata(tags)
                            except Exception:
                                logger.error("Did not manage to add tags")
                return (True, result)
        except Exception as ex:  # FileNotFoundError does not exist in 2.7
            # catch file not found and add an HTTP status code to return in notification to IoT Hub
            ex.status_code = 404
            return (False, ex)

    def _upload_sample(self, storage_info, file_name, tags, should_attach_metadata):
        # Upload to blob
        success, result = self._store_blob(storage_info, file_name, tags, should_attach_metadata)

        if success == True:
            logger.debug("Upload succeeded. Result is: " + str(result))
            self.device_client.notify_blob_upload_status(
                storage_info["correlationId"], True, 200, "OK: {}".format(file_name)
            )
            return True

        else:
            # If the upload was not successful, the result is the exception object
            logger.warn("Upload failed. Result is: " + str(result))

            self.device_client.notify_blob_upload_status(
                storage_info["correlationId"], False, result.status_code, str(result)
            )
            return False

    def upload_files(self, blob_storage_folder, abs_path, tags, should_attach_metadata):
        if self.is_device_connected is False:
            logger.warn("Device is not connected to IoT Hub")
            return False

        if os.path.isdir(abs_path):
            files = glob.glob(abs_path + "/*")
        elif os.path.isfile(abs_path):
            files = [abs_path]
        else:
            files = []

        for file in files:
            blob_name = blob_storage_folder + os.sep + os.path.basename(file)
            storage_info = self.device_client.get_storage_info_for_blob(blob_name)
            try:
                return self._upload_sample(storage_info, file, tags, should_attach_metadata)
            except Exception as ex:
                logger.error("Error uploading file: " + str(ex))
                return False

    def is_device_connected(self):
        return self.device_client.connected is True

    def upload_any_file(self, blob_storage_location, src_path, api_url, remove_after_upload=True):
        src_file_path = str(src_path)
        was_upload_successful = False
        if (self.is_device_connected() is True) :
            blob_folder = str(blob_storage_location)
            was_upload_successful = self.upload_files(blob_folder, src_file_path, None, False)

        if was_upload_successful is True :
            if remove_after_upload:
                try:
                    os.remove(src_file_path)
                except OSError as e:
                    logger.debug("Could not remove file " + src_file_path)
                return True
            else:
                logger.warn("upload files failed for " + src_file_path)
        return False
