import logging
from os.path import join, exists
from datetime import datetime, timedelta

global logger
logger = logging.getLogger('daemon.' + __name__)

class Timer(object):
    def __init__(self, path, weeks, days, hours, minutes):
        self.filename = join(path, 'timestamp.txt')
        self.time_format = '%Y%m%d_%H:%M:%S'
        self.__set_time_interval(weeks, days, hours, minutes)
        start_time = self.__get_start_time_from_file()
        self.__set_time_period(start_time)

    def reset(self):
        self.__set_time_period(datetime.now())
        self.__write_start_time_to_file(self.start_time)

    def __set_time_period(self, start_time):
        self.start_time = start_time
        self.__update_end_time()

    def __set_time_interval(self, weeks, days, hours, minutes):
        self.interval = timedelta(weeks=weeks, days=days, hours=hours, minutes=minutes)

    def __update_end_time(self):
        if self.start_time is not None:
            self.end_time = self.start_time + self.interval
            logger.info('Next automatic recording: ' + str(self.end_time))
        else:
            self.end_time = None

    def update_schedule(self, weeks, days, hours, minutes):
        self.__set_time_interval(weeks, days, hours, minutes)
        self.__update_end_time()

    def __get_start_time_from_file(self):
        if not exists(self.filename):
            logger.info("Previous recording time was not found")
            return None
        with open(self.filename, 'r') as fp:
            time_str = fp.readline().strip('\n')
            try:
                return datetime.strptime(time_str, self.time_format)
            except StandardError as e:
                logger.error("Unable to parse: ", e)
                return None

    def __write_start_time_to_file(self, time):
        time_str = time.strftime(self.time_format)
        with open(self.filename, 'w') as f:
            f.write(time_str)

    def non_blocking_wait(self):
        if self.end_time is None:
            return True
        return self.end_time < datetime.now()
