import requests
import os
import json
import uuid
import threading
from enum import Enum
import subprocess
import time
import datetime
import logging

class Download_status(Enum):
    IDLE = 'IDLE'
    DOWNLOADING = 'DOWNLOADING'
    DOWNLOAD_COMPLETE = 'DOWNLOAD_COMPLETE'
    FAILED = 'FAILED'

class RemoteUpdateHandler:
    def __init__(self, device_client_instance):
        global logger
        logger = logging.getLogger('daemon.' + __name__)
        self.device_client = device_client_instance
        self.processStateFileName = "download-update-process-state.json"
        self._create_remote_update_process_state_file_if_not_exists()
        self.state = self._get_state_from_file()
        self.rt = None
        self._reset()


    def start_download_update(self, file_url, file_name, file_size, target_version, override):
        logger.debug("file_url: " + str(file_url))
        logger.debug("file_name: " + str(file_name))
        logger.debug("file_size: " + str(file_size))
        logger.debug("target_version: " + str(target_version))
        logger.debug("override: " + str(override))
        current_status = self._get_status()
        if current_status != Download_status.IDLE.value and current_status != Download_status.FAILED.value:
            return DownloadFileHandlerResponse(self._get_status(), self._get_correlation_id(), 409,
                                               "Download is already running")

        if not file_url.startswith("https://s3-eu-west-1.amazonaws.com/ur-support-site/"):
            raise ValueError("The provided url is not recognized as a UR IoT hub domain.")

        try:
            path = self.get_target_version_update_path(target_version)
            logger.debug("Found path: '" + str(path) + "' for version: " + str(target_version))
        except Exception:
            path = None

        remote_update_directory_path = self._get_suitable_remote_update_file_directory_path(file_size)

        if override:
            #we did not find a path
            if path is None:
                if remote_update_directory_path is None:
                    raise Exception("Not enough disk space available")

            #we found a path
            else:
                logger.debug("deleting file from override")
                self.remove_old_update(path)
                remote_update_directory_path = self._get_suitable_remote_update_file_directory_path(file_size)

        else:
            if remote_update_directory_path is None:
                raise Exception("Not enough disk space available")

            #we did not find a path
            if path is None:
                path = remote_update_directory_path + file_name

            if "/remote_update_file/" in str(path):
                logger.debug("Own path")
                logger.debug("Path exists: " + str(self.file_name_already_exists(path)))
                if self.file_name_already_exists(path):
                    logger.debug("OS path size: " + str(os.path.getsize(path)))
                    logger.debug("Different size: " + str(os.path.getsize(path) != file_size))
                    if str(os.path.getsize(path)) != str(file_size):
                        self.remove_old_update(path)
                    else:
                        raise ValueError("Update already exists on disk.")
            else:
                validation_result = self.util_tool_is_target_version_valid(path)
                if not validation_result.success:
                    self.remove_old_update(path)
                else:
                    raise ValueError("Update already exists on disk.")


        self._update_correlation_id(self._generate_uuid())
        abs_save_path = remote_update_directory_path + file_name

        x = threading.Thread(target=self._start_download_process_flow, args=(file_url,abs_save_path,file_size))
        x.daemon = True
        x.start()

        return DownloadFileHandlerResponse(self._get_status(), self._get_correlation_id(), 200, "Download started")

    def validate_update_request_for_target_version(self, target_version):
        current_status = self._get_status()
        if current_status != Download_status.IDLE.value and current_status != Download_status.FAILED.value:
            raise Exception("Update is being downloaded. Please wait for it to complete before starting an update")

        path = self.get_target_version_update_path(target_version)
        validation_result = self.util_tool_is_target_version_valid(path)
        if not validation_result.success:
            raise Exception(validation_result.message)

    def try_start_update(self, target_version):
        path = self.get_target_version_update_path(target_version)
        x = threading.Thread(target=self._start_update_process_flow, args=(path,))
        x.daemon = True
        x.start()


    def get_target_version_update_path(self, target_version):

        try:
            byte_output = subprocess.check_output(['sh', './apply_urup_update.sh', '-l', str(target_version)])
            output = byte_output.decode('UTF-8').rstrip()

            if "ERROR" in output:
                raise Exception("Unable to locate the update file on the robot")
            path = str(output)
            logger.debug("Path found: " + str(path))

            return path
        except subprocess.CalledProcessError as e:
            logger.error(str(e))
            raise Exception(str(e))

    def util_tool_is_target_version_valid(self, path):
        try:
            byte_output = subprocess.check_output(['sh', './apply_urup_update.sh', '-v', path])
            output = byte_output.decode('UTF-8').rstrip()

            if not "Validation OK" in str(output):
                logger.debug("Is valid: " + str(output))
                return UtilToolValidationResult(False, str(output))

            return UtilToolValidationResult(True, "")
        except Exception as e:
            logger.error("Is valid: " + str(e))
            return UtilToolValidationResult(False, str(e))

    def trigger_update(self, path):
        byte_output = subprocess.check_output(['sh', './apply_urup_update.sh', '-u', path])
        output = byte_output.decode('UTF-8').rstrip()

        logger.debug(str(output))

    def _get_usb_path(self, required_size):
        arr = ['usbdisk', 'usbdisk_0', 'usbdisk_1', 'usbdisk_2', 'usbdisk_3']
        usb_found = False
        for n in arr:
            path = "/programs/" + str(n)
            if os.path.exists(path):
                usb_found = True
                st = os.statvfs(path)
                available_space = st.f_bavail * st.f_frsize
                if long(required_size) < long(available_space):
                    return path

        if usb_found:
            return None
        raise Exception("No USB Disk found")

    def file_name_already_exists(self, path):
        return os.path.exists(path)

    def remove_old_update(self, path):

        try:
            if os.path.isfile(path) or os.path.islink(path):
                os.unlink(path)

        except Exception as e:
            print('Failed to delete %s. Reason: %s' % (path, e))


    def _start_download_process_flow(self, file_url,file_name,file_size):
        try:

            self._update_status(Download_status.DOWNLOADING.value)
            self._send_status_update_to_hub()
            self.start_sending_timed_updates()

            if self.is_robot():
                subprocess.call('mount -o remount,async ' + str(self._get_usb_path(file_size)), shell=True)

            with open(file_name, "wb") as f:

                response = requests.get(file_url, stream=True, timeout=30)
                total_length = response.headers.get('content-length')

                if total_length is None: # no content length header
                    f.write(response.content)
                else:
                    dl = 0
                    total_length = int(total_length)
                    for data in response.iter_content(chunk_size=4096):
                        dl += len(data)
                        f.write(data)
                        percentage_complete = int(100 * dl / total_length)
                        self._update_percentage_complete(percentage_complete)


            self._update_percentage_complete(100)
            self.stop_sending_timed_updates()
            self._update_status(Download_status.DOWNLOAD_COMPLETE.value)
            self._send_status_update_to_hub()

            self._reset()
        except Exception as e:
            self.stop_sending_timed_updates()
            self._update_status(Download_status.FAILED.value)
            self._update_message(str(e))
            self._update_percentage_complete(0)
            self._send_status_update_to_hub()
            self._reset()

    def _start_update_process_flow(self, path):
        try:
            logger.debug("Triggering start update with target version " + str(path))
            time.sleep(1)

            self.trigger_update(path)

        except Exception as e:
            logger.error("Could not trigger update because of following error: " + str(e))

    def start_sending_timed_updates(self):
        if not self.rt is None:
            self.rt.stop()
            self.rt = None

        self.rt = RepeatedTimer(5, self._send_status_update_to_hub) # it auto-starts, no need of rt.start()

    def stop_sending_timed_updates(self):
        if not self.rt is None:
            self.rt.stop()

    def _send_status_update_to_hub(self):
        self._send_message_to_hub(
            '{ "type" : "DownloadUpdateRemotelyStatus", "state" : ' + json.dumps(self.state) + ' }')

    def _send_message_to_hub(self, message):
        try:
            self.device_client.send_message(message)
            return True
        except Exception as e:
            logger.error("Failed to send message : " + message + " to hub :" + e)
            return False

    def _create_remote_update_process_state_file_if_not_exists(self):
        path = self._get_state_file_path()
        if not os.path.exists(path):
            os.makedirs(path)

        self.processStateFileNameFullPath = path + self.processStateFileName

        if os.path.exists(self.processStateFileNameFullPath):
            with open(self.processStateFileNameFullPath, "r") as f:
                line = f.read()
            if line == "":
                d = json.loads('{"status": "'+Download_status.IDLE.value+'", "correlationId" : "", "percentageComplete" : 0, "message" : "" }')
                self._write_to_status_file(json.dumps(d), path + self.processStateFileName)
        else:
            d = json.loads('{"status": "'+Download_status.IDLE.value+'", "correlationId" : "", "percentageComplete" : 0, "message" : "" }')
            self._write_to_status_file(json.dumps(d), path + self.processStateFileName)

    def _write_to_status_file(self, data, filepath):
        with open(filepath, 'w') as f:
            f.write(data)

    def _write_to_log_file(self, log_message):

        if self.is_robot():
            log_file_path = "/data/myUR/remote_update.log"
        else:
            log_file_path = "/tmp/remote_update_file/remote_update.log"

        now = datetime.datetime.now()
        with open(log_file_path, 'a+') as f:
            f.write(str(now) + ": " + log_message + "\n")


    def _get_state_from_file(self):
        with open(self.processStateFileNameFullPath, "r") as f:
            state = json.loads(f.read())
        return state

    def is_robot(self):
        return os.path.exists("/root/ur-serial")

    def _get_suitable_remote_update_file_directory_path(self, required_size):
        if os.path.exists("/root/ur-serial"):
            usb_path = self._get_usb_path(required_size)

            if usb_path is None:
                return None

            directory = usb_path + "/remote_update_file/"
        else:
            directory = "/tmp/remote_update_file/"

        if not os.path.exists(directory):
            os.makedirs(directory)

        return directory

    def _get_state_file_path(self):
        if os.path.exists("/root/ur-serial"):
            directory = "/data/myUR/"
        else:
            directory = "/tmp/"
        return directory

    def _get_correlation_id(self):
        return self.state["correlationId"]

    def _get_status(self):
        return self.state["status"]

    def _update_correlation_id(self, correlationId):
        self.state["correlationId"] = correlationId
        self._write_to_status_file(json.dumps(self.state), self.processStateFileNameFullPath)

    def _update_status(self, status):
        self.state["status"] = status
        self._write_to_status_file(json.dumps(self.state), self.processStateFileNameFullPath)

    def _update_message(self, message):
        self.state["message"] = message
        self._write_to_status_file(json.dumps(self.state), self.processStateFileNameFullPath)

    def _get_percentage_complete(self):
        return self.state["percentageComplete"]

    def _update_percentage_complete(self, complete):
        self.state["percentageComplete"] = complete
        self._write_to_status_file(json.dumps(self.state), self.processStateFileNameFullPath)

    def _generate_uuid(self):
        return str(uuid.uuid4())

    def _reset(self):
        self._update_status(Download_status.IDLE.value)
        self._update_correlation_id("")
        self._update_percentage_complete(0)
        self._update_message("")
        self.rt = None

class DownloadFileHandlerResponse:
    def __init__(self, current_status, correlation_id, status_code, message):
        self.currentStatus = current_status
        self.correlationId = correlation_id
        self.statusCode = status_code
        self.message = message


class RepeatedTimer(object):
    def __init__(self, interval, function, *args, **kwargs):
        self._timer     = None
        self.interval   = interval
        self.function   = function
        self.args       = args
        self.kwargs     = kwargs
        self.is_running = False
        self.start()

    def _run(self):
        self.is_running = False
        self.start()
        self.function(*self.args, **self.kwargs)

    def start(self):
        if not self.is_running:
            self._timer = threading.Timer(self.interval, self._run)
            self._timer.start()
            self.is_running = True

    def stop(self):
        self._timer.cancel()
        self.is_running = False

class UtilToolValidationResult:
    def __init__(self, success, message):
        self.success = success
        self.message = message