import time
import shutil
import os
import logging
import fnmatch
from watchdog.observers import Observer
from watchdog.events import PatternMatchingEventHandler
import threading

program_dir = '/programs/'
if os.path.exists("/root/ur-serial") is True:
    backup_dir = '/data/myUR/.backups/'
else :
    backup_dir = '/tmp/myUR/.backups/'

upload_to_blob_service = None
global logger
logger = logging.getLogger('daemon.' + __name__)
class ProgramDirectoryFilesBackupService :
    def __init__(self):
        self.watcher = Watcher()
        self._create_backup_dir()
        self._start_services()

    def _create_backup_dir(self):
        if not os.path.exists(backup_dir):
            os.makedirs(backup_dir)

    def set_connection_details(self, upload_to_blob_handler):
        global upload_to_blob_service
        upload_to_blob_service = upload_to_blob_handler

    def on_connection_state_change(self):
        #logger.debug("Connected to IoT Hub, checking for pending backups")
        self._bunch_upload_backups_when_connected(backup_dir)

    def _bunch_upload_backups_when_connected(self, backup_dir):
        all_files = os.listdir(backup_dir)
        for file in all_files:
            file_abs_path = os.path.join(backup_dir, file)
            if os.path.isdir(file_abs_path):
                self._bunch_upload_backups_when_connected(os.path.join(backup_dir, file))
            elif os.path.isfile(file_abs_path) and (file.endswith(".urp") or file.endswith(".installation")):
                patch_list = []
                for matched_file in fnmatch.filter(all_files,   file + '.modified_*') :
                    patch_list.append(os.path.join(backup_dir, matched_file))
                self._upload_all_versions_for_file(file, file_abs_path, patch_list)

    def _upload_all_versions_for_file(self, file, file_abs_path, modified_file_list):
        logger.info("Uploading all versions for file: " + file)
        if uploadBackupToBlob(file_abs_path, True) is False:
            return
        parent_dir = os.path.dirname(file_abs_path)
        new_parent = os.path.join(parent_dir, ".tmp")
        if not os.path.exists(new_parent):
            os.makedirs(new_parent)
        for modified_file in modified_file_list:
            copied_file_path_to_tmp = os.path.join(new_parent, file)
            shutil.copy2(modified_file, copied_file_path_to_tmp)
            os.remove(modified_file)
            if uploadBackupToBlob(copied_file_path_to_tmp, True) is False:
                return
        try :
            os.rmdir(new_parent)
        except OSError as e:
            logger.warn("Could not remove directory as it is not empty ")

    def _start_services(self):
        patterns1=["*.urp", "*.installation"]
        create_patch_handler = ProgramDirectoryEventHandler(patterns1)
        self.watcher.schedule_observer(create_patch_handler, program_dir)

        upload_backups_handler_on_connected = BackupsDirectoryEventHandler(patterns1)
        self.watcher.schedule_observer(upload_backups_handler_on_connected, backup_dir)
        threading.Thread(target=self.watcher.run).start()

    def stop_services(self):
        self.watcher.stop()


class Watcher:
    def __init__(self):
        self.observer = Observer()
        self.observer.daemon = True

    def run(self):
        self.observer.start()
        logger.info("Started the watcher for path ... " )
        try:
            while True:
               time.sleep(1)
        except Exception as e:
            logger.error("Error: " + str(e))
            logger.warn("Stopping the watcher unexpectedly..." )
            self.observer.stop()
        self.observer.join()

    def schedule_observer(self, event_handler, path, recursive=True):
        self.observer.schedule(event_handler, path, recursive)
    def stop(self):
        logger.info("Stopping the watcher gracefully..." + self.path)
        self.observer.stop()
        self.observer.join()

class ProgramDirectoryEventHandler(PatternMatchingEventHandler):
    def __init__(self, patterns_list):
        logger.info("ProgramDirectoryEventHandler initialized")
        self.files_to_ignore = set([])
        # Set the patterns for PatternMatchingEventHandler
        PatternMatchingEventHandler.__init__(
            self,
            patterns=patterns_list,
            ignore_patterns=['*/.tmp/*'],
            ignore_directories=False,
            case_sensitive=True
        )

    def on_closed(self, event):
        try :
            self._create_backup(event)
        except Exception as e:
            logger.error("Error while creating backups: " + str(e))

    def _create_backup(self, event):
        file_to_copy = event.src_path
        file_parent_dir, changed_file_name = os.path.split(file_to_copy)

        restore_file = os.path.normpath(program_dir + ".restore_" + changed_file_name)
        if os.path.exists(restore_file):
            logger.debug("Ignoring file backup as it is a restored file")
            os.remove(restore_file)
            return

        new_target_directory_path=os.path.normpath(backup_dir + file_parent_dir)
        new_destination_file = os.path.join(new_target_directory_path, changed_file_name)

        if not os.path.exists(new_target_directory_path):
            os.makedirs(new_destination_file.replace(changed_file_name, ""))

        if not os.path.exists(new_destination_file):
            shutil.copy2(file_to_copy, new_destination_file)
        else :
            logger.debug("File already exists in backup directory, creating versions... ")
            existing_list = []
            all_existing_sorted_files = self._sorted_directory_listing_by_creation_time_with_os_listdir(new_target_directory_path)
            for matched_file in fnmatch.filter(all_existing_sorted_files,   changed_file_name + '.modified_*') :
                existing_list.append(matched_file)
            if len(existing_list) > 8:
                src_file = os.path.join(new_target_directory_path, existing_list[0])
                dest_file = os.path.join(new_target_directory_path, changed_file_name)
                os.rename(src_file, dest_file)
            modified_file_name = new_destination_file + ".modified_" + str(time.time())
            shutil.copy2(file_to_copy, modified_file_name)

    def _sorted_directory_listing_by_creation_time_with_os_listdir(self, directory):
        def get_creation_time(item):
            item_path = os.path.join(directory, item)
            return os.path.getctime(item_path)

        items = os.listdir(directory)
        sorted_items = sorted(items, key=get_creation_time)
        return sorted_items

class BackupsDirectoryEventHandler(PatternMatchingEventHandler):
    def __init__(self, patterns_list):
        # Set the patterns for PatternMatchingEventHandler
        PatternMatchingEventHandler.__init__(
            self,
            patterns=patterns_list,
            ignore_patterns=['*/.tmp/*'],
            ignore_directories=False,
            case_sensitive=True
        )

    def on_closed(self, event):
        logger.debug("Back up triggered for " + event.src_path)
        abs_file_path = event.src_path
        uploadBackupToBlob(abs_file_path)


def uploadBackupToBlob(abs_file_path, remove_after_upload=True):
    if (upload_to_blob_service is not None and upload_to_blob_service.is_device_connected() is True) :
        logger.debug("Uploading backup to blob storage for file: " + abs_file_path)
        blob_folder = os.path.dirname(abs_file_path)
        blob_folder = blob_folder.replace(backup_dir, "")
        blob_folder = blob_folder.replace("/.tmp", "")
        was_upload_successful = upload_to_blob_service.upload_files(blob_folder, abs_file_path, None, False)
        if was_upload_successful is True :
            if remove_after_upload:
                logger.debug("Removing file after successful upload: " + abs_file_path)
                os.remove(abs_file_path)
                logger.debug("Deleting folder if empty: " + blob_folder.split("/")[0])
                try:
                    os.rmdir(blob_folder.split("/")[0])
                except OSError as e:
                    logger.debug("Could not remove directory as it is not empty ")
            return True
        else:
            logger.warn("uploadBackupToBlob failed for " + abs_file_path)
            return False
    return False


