import os
import json
import uuid
import threading
import subprocess
import glob
from enum import Enum
from datetime import datetime
import logging

global logger
logger = logging.getLogger('daemon.' + __name__)

class Generate_File_State(Enum):
    IDLE = 'IDLE'
    EXTRACTING = 'EXTRACTING'
    UPLOADING = 'UPLOADING'
    UPLOAD_COMPLETE = 'UPLOAD_COMPLETE'
    FAILED = 'FAILED'

class GenerateFileHandler:

    def __init__(self, message_handler, file_generator, state_file, blob_storage_folder, hub_status):
        self.message_handler = message_handler
        self.blob_uploader = message_handler.upload_to_blob_handler
        self.processStateFileName = state_file
        self.processStateFileNameFullPath = ""  # is set through create_support_file_process_state_file_if_not_exists
        self._create_support_file_process_state_file_if_not_exists()
        self.state = self._get_state_from_file()
        self._reset()
        self.file_generator = file_generator
        self.blob_storage_folder = blob_storage_folder
        self.hub_status = hub_status

    def generate_file(self, tags):
        current_status = self._get_status()
        if current_status != Generate_File_State.IDLE.value and current_status != Generate_File_State.FAILED.value:
            return GenerateFileHandlerResponse(self._get_status(), self._get_correlation_id(), 409,
                                              "Process is already running")

        self._update_correlation_id(self._generate_uuid())

        x = threading.Thread(target=self._start_process_flow, args=(tags,))
        x.daemon = True
        x.start()

        return GenerateFileHandlerResponse(self._get_status(), self._get_correlation_id(), 200, "Extraction started")

    def _start_process_flow(self, tags):
        try:
            self._update_status(Generate_File_State.EXTRACTING.value)
            self._send_status_update_to_hub()

            correlation_id = self._get_correlation_id()
            fileName = correlation_id
            if str(tags["Name"]) == "UrServiceFile" :
                fileName = "ur_service_" + str(tags["SerialNumber"]) + '_' + datetime.now().strftime('%Y-%m-%d_%H-%M-%S')
                self._update_file_name(fileName)
                tags["correlation_id"] = correlation_id # correlation_id is used for handling H&W processing in Customer portal backend later.

            file_path = self.file_generator.generate(self._get_upload_directory(), fileName)
            tags = self.file_generator.add_tags(tags)

            self._update_file_size(os.path.getsize(file_path))
            self._update_status(Generate_File_State.UPLOADING.value)
            self._send_status_update_to_hub()

            self.blob_uploader.upload_files(self.blob_storage_folder, file_path, tags, True)
            self._update_status(Generate_File_State.UPLOAD_COMPLETE.value)
            self._send_status_update_to_hub()

            self._reset()
            self._delete_file()

        except Exception as e:
            self._update_status(Generate_File_State.FAILED.value)
            self._update_message(str(e))
            self._send_status_update_to_hub()
            self._reset()

    def _send_status_update_to_hub(self):
        logger.debug("send: " + json.dumps(self.state))
        self.message_handler.send_message_to_hub(
            '{ "type": "' + self.hub_status + '", "state": ' + json.dumps(self.state) + ' }')

    def _reset(self):
        self._update_status(Generate_File_State.IDLE.value)
        self._update_correlation_id("")
        self._update_file_size(0)
        self._update_message("")
        self._update_file_name("")

    def _create_support_file_process_state_file_if_not_exists(self):
        path = self._get_state_file_path()
        if not os.path.exists(path):
            os.makedirs(path)

        self.processStateFileNameFullPath = path + self.processStateFileName

        if not os.path.exists(self.processStateFileNameFullPath):
            d = json.loads('{"status": "' + Generate_File_State.IDLE.value + '", "correlationId" : "", "supportFileSize" : 0, "message" : "" }')
            self._write_to_status_file(json.dumps(d), self.processStateFileNameFullPath)

    def _write_to_status_file(self, data, filepath):
        with open(filepath, 'w') as f:
            f.write(data)

    def _get_state_from_file(self):
        with open(self.processStateFileNameFullPath, "r") as f:
            state = json.loads(f.read())
        return state

    def _delete_file(self):
        for file in glob.glob(self._get_upload_directory() + "*.zip"):
            if os.path.exists(file):
                os.remove(file)

    def _get_state_file_path(self):
        if os.path.exists("/root/ur-serial"):
            directory = "/data/myUR/"
        else:
            directory = "/tmp/myUR/"
        return directory

    def _get_upload_directory(self):
        return self._get_state_file_path()

    def _get_correlation_id(self):
        return self.state["correlationId"]

    def _get_status(self):
        return self.state["status"]

    def _update_correlation_id(self, correlationId):
        self.state["correlationId"] = correlationId
        self._write_to_status_file(json.dumps(self.state), self.processStateFileNameFullPath)

    def _update_status(self, status):
        self.state["status"] = status
        self._write_to_status_file(json.dumps(self.state), self.processStateFileNameFullPath)

    def _update_message(self, message):
        self.state["message"] = message
        self._write_to_status_file(json.dumps(self.state), self.processStateFileNameFullPath)

    def _update_file_size(self, size):
        self.state["supportFileSize"] = size
        self._write_to_status_file(json.dumps(self.state), self.processStateFileNameFullPath)

    def _update_file_name(self, fileName):
        self.state["fileName"] = fileName
        self._write_to_status_file(json.dumps(self.state), self.processStateFileNameFullPath)

    def _generate_uuid(self):
        return str(uuid.uuid4())


class GenerateFileHandlerResponse:
    def __init__(self, current_status, correlation_id, status_code, message):
        self.currentStatus = current_status
        self.correlationId = correlation_id
        self.statusCode = status_code
        self.message = message
