import requests
import os
import json
import uuid
import threading
from enum import Enum
import logging
import datetime

class Download_status(Enum):
    IDLE = 'IDLE'
    DOWNLOADING = 'DOWNLOADING'
    DOWNLOAD_COMPLETE = 'DOWNLOAD_COMPLETE'
    FAILED = 'FAILED'

class FileDownloader:
    def __init__(self, device_client, download_type, process_state_file, log_file_name="remote_update.log"):
        self.download_request_type = download_type
        self.device_client = device_client
        global logger
        logger = logging.getLogger('daemon.' + __name__)
        if self.is_robot():
            self.myur_base_path = "/data/myUR/"
        else:
            self.myur_base_path = "/tmp/"

        self.log_file_path = self.myur_base_path + log_file_name

        self.processStateFileName = process_state_file
        self._create_remote_update_process_state_file_if_not_exists()
        self.state = self._get_state_from_file()
        self.rt = None
        self._reset()

    def start_download_update(self, file_url, abs_save_path):
        current_status = self._get_status()
        if current_status != Download_status.IDLE.value and current_status != Download_status.FAILED.value:
            return DownloadFileHandlerResponse(self._get_status(), self._get_correlation_id(), 409,
                                               "Download is already running")

        self._update_correlation_id(self._generate_uuid())

        x = threading.Thread(target=self._start_download_process_flow, args=(file_url,abs_save_path))
        x.daemon = True
        x.start()

        return DownloadFileHandlerResponse(self._get_status(), self._get_correlation_id(), 200, "Download started")

    def file_name_already_exists(self, path):
        return os.path.exists(path)

    def remove_old_update(self, path):

        try:
            if os.path.isfile(path) or os.path.islink(path):
                os.unlink(path)

        except Exception as e:
            logger.error('Failed to delete %s. Reason: %s' % (path, e))


    def _start_download_process_flow(self, file_url,file_name):
        try:

            self._update_status(Download_status.DOWNLOADING.value)
            self._send_status_update_to_hub()
            self.start_sending_timed_updates()

            #if self.is_robot():
                #subprocess.call('mount -o remount,async ' + str(self._get_usb_path(file_size)), shell=True)

            with open(file_name, "wb") as f:

                response = requests.get(file_url, stream=True, timeout=30)
                total_length = response.headers.get('content-length')

                if total_length is None: # no content length header
                    f.write(response.content)
                else:
                    dl = 0
                    total_length = int(total_length)
                    for data in response.iter_content(chunk_size=4096):
                        dl += len(data)
                        f.write(data)
                        percentage_complete = int(100 * dl / total_length)
                        self._update_percentage_complete(percentage_complete)


            self._update_percentage_complete(100)
            self.stop_sending_timed_updates()
            self._update_status(Download_status.DOWNLOAD_COMPLETE.value)
            self._send_status_update_to_hub()

            self._reset()
        except Exception as e:
            self.stop_sending_timed_updates()
            self._update_status(Download_status.FAILED.value)
            self._update_message(str(e))
            self._update_percentage_complete(0)
            self._send_status_update_to_hub()
            self._reset()

    def start_sending_timed_updates(self):
        if not self.rt is None:
            self.rt.stop()
            self.rt = None

        self.rt = RepeatedTimer(5, self._send_status_update_to_hub) # it auto-starts, no need of rt.start()

    def stop_sending_timed_updates(self):
        if not self.rt is None:
            self.rt.stop()

    def _send_status_update_to_hub(self):
        try:
            self.device_client.send_message('{ "type" : ' + self.download_request_type + ', "state" : ' + json.dumps(self.state) + ' }')
            return True
        except Exception as e:
            logger.error("Failed to send message : " + message + " to hub :" + e)
            return False

    def _create_remote_update_process_state_file_if_not_exists(self):
        self.processStateFileNameFullPath = self.myur_base_path + self.processStateFileName

        if os.path.exists(self.processStateFileNameFullPath):
            with open(self.processStateFileNameFullPath, "r") as f:
                line = f.read()
            if line == "":
                d = json.loads('{"status": "' + Download_status.IDLE.value+'", "correlationId" : "", "percentageComplete" : 0, "message" : "" }')
                self._write_to_status_file(json.dumps(d),  self.processStateFileNameFullPath)
        else:
            d = json.loads('{"status": "' + Download_status.IDLE.value+'", "correlationId" : "", "percentageComplete" : 0, "message" : "" }')
            self._write_to_status_file(json.dumps(d), self.processStateFileNameFullPath)

    def _write_to_status_file(self, data, filepath):
        with open(filepath, 'w') as f:
            f.write(data)

    def _write_to_log_file(self, log_message):
        now = datetime.datetime.now()
        with open(self.log_file_path, 'a+') as f:
            f.write(str(now) + ": " + str(self.download_request_type).upper() + " : " +  log_message + "\n")


    def _get_state_from_file(self):
        with open(self.processStateFileNameFullPath, "r") as f:
            state = json.loads(f.read())
        return state

    def is_robot(self):
        return os.path.exists("/root/ur-serial")

    def _get_correlation_id(self):
        return self.state["correlationId"]

    def _get_status(self):
        return self.state["status"]

    def _update_correlation_id(self, correlationId):
        self.state["correlationId"] = correlationId
        self._write_to_status_file(json.dumps(self.state), self.processStateFileNameFullPath)

    def _update_status(self, status):
        self.state["status"] = status
        self._write_to_status_file(json.dumps(self.state), self.processStateFileNameFullPath)

    def _update_message(self, message):
        self.state["message"] = message
        self._write_to_status_file(json.dumps(self.state), self.processStateFileNameFullPath)

    def _get_percentage_complete(self):
        return self.state["percentageComplete"]

    def _update_percentage_complete(self, complete):
        self.state["percentageComplete"] = complete
        self._write_to_status_file(json.dumps(self.state), self.processStateFileNameFullPath)

    def _generate_uuid(self):
        return str(uuid.uuid4())

    def _reset(self):
        self._update_status(Download_status.IDLE.value)
        self._update_correlation_id("")
        self._update_percentage_complete(0)
        self._update_message("")
        self.rt = None

class DownloadFileHandlerResponse:
    def __init__(self, current_status, correlation_id, status_code, message):
        self.currentStatus = current_status
        self.correlationId = correlation_id
        self.statusCode = status_code
        self.message = message


class RepeatedTimer(object):
    def __init__(self, interval, function, *args, **kwargs):
        self._timer     = None
        self.interval   = interval
        self.function   = function
        self.args       = args
        self.kwargs     = kwargs
        self.is_running = False
        self.start()

    def _run(self):
        self.is_running = False
        self.start()
        self.function(*self.args, **self.kwargs)

    def start(self):
        if not self.is_running:
            self._timer = threading.Timer(self.interval, self._run)
            self._timer.start()
            self.is_running = True

    def stop(self):
        self._timer.cancel()
        self.is_running = False

class UtilToolValidationResult:
    def __init__(self, success, message):
        self.success = success
        self.message = message