#!/usr/bin/env bash

function get_urup_to_be_installed {
   if [[ -z "$1" ]]; then
         echo "ERROR: Missing software version, exiting .. "
         return 1
   fi

   local look_for_version_string="ver. $1"
   while read -r line
   do
      if [[ "$line" == *"$look_for_version_string"* ]]; then
         local path_to_urup_to_install=$(echo "$line" | cut -d' ' -f1)
         echo "$path_to_urup_to_install"
         return
     fi
    done <<< "$(urupdate --list-updates)"
    echo "ERROR: No installable found for Software Version $1"
    return 1
 }


# Send command to Dashboard Server
function dashboard_command {
    # send dashboard command to the server and "quit" to return, then cut away the first and last line and preserve only
    # output related to the command.
    (
        echo "$1"
        echo "quit"
    ) | nc 127.0.0.1 29999 | tail -n +2 | head -1
}

function verify_program_is_not_running {
  result=$(dashboard_command "programState")
  stopped="STOPPED"
  if [[ $result == *"$stopped"* ]]; then
    return 0
  else
    return 1
  fi
}

function power_off_robot {
  dashboard_command "power off"
}

function is_robot_powered_off {
  result=$(dashboard_command "robotmode")
  power_off="POWER_OFF"
  if [[ $result == *"$power_off"* ]]; then
    return 0
  else
     return 1
  fi
}

function verify_requirements {
  local urup_file="$1"
  if [[ -z "$urup_file" ]]; then
    echo "ERROR: Missing path to urup file to verify, exiting .. "
    return 1
  fi
  urupdate --verify-requirements "$urup_file"
  if [[ $? != 0 ]]; then
     echo "ERROR: Invalid urup file"
     return 1
  fi
  verify_program_is_not_running
  if  [[ $? != 0 ]] ; then
     echo "ERROR: Program is playing"
     return 1
  fi
  echo "Validation OK"
}

function install_updates {
  if [[ -z "$1" ]]; then
      echo "ERROR: Missing file to install"
      return 1
  fi
  power_off_robot
  is_robot_powered_off
  if [[ $? == 0 ]]  ; then
    echo "urupdate --update $1"
    urupdate --update "$1"
  else
    echo "ERROR: Unable to update, robot is not powered off"
    return 1;
  fi
  echo "Initialising update ...."
}

function print_usage() {
  echo "Error: Invalid Invocation
        Usage: [-v or --verify FILE | -l or --list SW_VERSION | -u or --update FILE]"
}

if [[ ("$1" != '-v') && ("$1" != '-l')  && ("$1" != '-u') && ("$1" != '--verify') && ("$1" != '--list') && ("$1" != '--update') ]]; then
  print_usage
  exit 1
fi

while [ $# -gt 0 ]; do
   case $1 in
  -v | --verify )
    verify_requirements "$2"
    break
    ;;
  -l | --list )
    get_urup_to_be_installed "$2"
    break
    ;;
  -u | --update )
    install_updates "$2"
    break
    ;;
   *)
   print_usage
   ;;
esac; done


exit 0