from os import makedirs, system, listdir, remove
from os.path import exists, join, isfile
from zipfile import ZipFile
import logging
import subprocess
import re
global logger
logger = logging.getLogger('daemon.' + __name__)

from FileGenerator import FileGenerator
from sherlock import dashboard_server, record

class UrServiceFileGenerator(FileGenerator):
    def generate(self, file_directory_path, correlation_id, wait_for_movement=False, recording_length=90):
        disk_space = self._get_available_disk_space(file_directory_path)
        if disk_space < 200:
            raise Exception("Insufficient disk space: " + str(disk_space) + " MB")

        tmp_path = self._make_ur_service_directory(file_directory_path)
        record_succeeded = self._record_realtime_data(tmp_path, wait_for_movement, recording_length)
        if not record_succeeded:
            raise Exception("RTDE data extraction failed")

        robot_config = self._trigger_robot_configuration_file(tmp_path)
        self._copy_installation_from_zip(robot_config, tmp_path)
        self._copy_program_and_script(tmp_path)
        self._copy_log_history(tmp_path)
        zip_path = self._create_zip_file(file_directory_path, tmp_path, correlation_id)
        self._cleanup(tmp_path)
        return zip_path

    def add_tags(self, tags):
        tags['UrServiceFileTrigger'] = 'Manual'
        return tags

    def _create_zip_file(self, output_path, input_path, name):
        zip_path = join(output_path, name + ".zip")

        if self._is_simulation():
            with ZipFile(zip_path, 'w') as zip_object:
                for f in listdir(input_path):
                    p = join(input_path, f)
                    if isfile(p):
                        zip_object.write(p, p.split('/')[-1])
        else:
            system('zip -qj ' + zip_path + ' ' + join(input_path, '*'))
        return zip_path

    def _cleanup(self, path):
        for f in listdir(path):
            file = join(path, f)
            if exists(file):
                remove(file)

    def _get_available_disk_space(self, path):
        output = subprocess.check_output(["df", path])
        line = str(output).splitlines()[-1] # extract last line
        line = re.sub(' +', ' ', line) # remove extra white space
        cols = line.split(' ') # split into array of values
        #total = float(cols[1]) / (2**10)
        #used = float(cols[2]) / (2**10)
        free = float(cols[3]) / (2**10)
        return free

    def _record_realtime_data(self, path, use_trigger, seconds):
        recording_file_name = 'realtimedata.csv'
        recording_path = join(path, recording_file_name)
        ip = '127.0.0.1'
        binary = True
        buffered = True
        frequency = 500
        samples = int(frequency * seconds)
        logger.debug("Record " + str(seconds) + " sec, wait for move: " + str(use_trigger))
        success = record.run(samples, binary, recording_path, frequency, buffered, ip, use_trigger)
        return success

    def _make_ur_service_directory(self, path):
        ur_service_path = join(path, "ur_service")
        if not exists(ur_service_path):
            makedirs(ur_service_path)

        return ur_service_path

    def _trigger_robot_configuration_file(self, destination_path):
        dashboard = dashboard_server.DashboardControl("127.0.0.1")
        dashboard.connect()
        result = dashboard.generate_robot_configuration_file()
        dashboard.disconnect()
        name = result.split(' ')[-1] + ".zip"
        report_path = join(self._get_flightrecord_dir(), name)
        destination_name = join(destination_path, name)
        self._copy_file(report_path, destination_name)
        return destination_name


    def _get_flightrecord_dir(self):
        if self._is_simulation():
            root_dir = "/ursim"
        else:
            root_dir = "/root"

        return join(root_dir, "flightreports")

    def _copy_log_history(self, path):
        if self._is_simulation():
            logs_path = "/ursim/log_history.txt"
        else:
            logs_path = "/root/log_history.txt"
        self._copy_file(logs_path, path)

    def _copy_program_and_script(self, path):
        dashboard = dashboard_server.DashboardControl("127.0.0.1")
        dashboard.connect()
        program_name = dashboard.get_loaded_program().encode(encoding='UTF-8').split(' ')[-1]
        dashboard.disconnect()
        script_name = program_name.replace('.urp', '.script')
        self._copy_file(script_name, path)
        self._copy_file(program_name, path)

    def _copy_installation_from_zip(self, report_path, destination_path):
        with ZipFile(report_path) as z:
            for name in z.namelist():
                split_path = name.split('/')
                if len(split_path) >= 2 and ".installation" in split_path[1]:
                    installation_name = name.split('/')[-1]
                    installation_path = join(destination_path, installation_name)
                    with open(installation_path, 'wb') as f:
                        f.write(z.read(name))
