import logging
import rtde
import rtde.csv_writer as csv_writer
import rtde.csv_binary_writer as csv_binary_writer
import recipe
import string
import time

global logger
logger = logging.getLogger('daemon.' + __name__)

def should_trigger(state):
    if (state.motor_command == 139 or state.motor_command == 0) and \
            (state.target_qd[0] != 0 or state.target_qd[1] != 0 or
             state.target_qd[2] != 0 or state.target_qd[3] != 0 or
             state.target_qd[4] != 0 or state.target_qd[5] != 0):
        return True
    return False

def prepare_alternative_header(header):
    new_header = []
    for s1 in header:
        s2 = string.replace(s1, "gear_torsion", "gt")
        new_header.append(s2)
    return new_header

def run(samples, binary, output, frequency, buffered, ip, use_trigger=True):
    try:
        con = rtde.RTDE(ip, 30004)
        try:
            con.connect()
        except Exception as e:
            con.disconnect()
            logger.error("Unable to connect " + str(e))
            return False

        abort_state = False
        if con.is_connected():
            # setup recipe based on controller version
            actual = con.get_controller_version()
            conf = None
            if actual[0] == 5 and actual[1] >= 9:
                conf = recipe.conf_5_9

            if conf is None:
                logger.error("Old controller version")
                con.disconnect()
                return False

            output_names = [f[0] for f in conf]
            output_types = [f[1] for f in conf]
            alternative_header = prepare_alternative_header(output_names)

            if not con.send_output_setup(output_names, output_types, frequency):
                logger.error("Unable to configure output")
                con.disconnect()
                return False

            # start data synchronization
            if not con.send_start():
                logger.error("Unable to start synchronization")
                con.disconnect()
                return False

            write_modes = "wb" if binary else "w"
            with open(output, write_modes) as csvfile:
                if binary:
                    writer = csv_binary_writer.CSVBinaryWriter(csvfile, alternative_header, output_types)
                else:
                    writer = csv_writer.CSVWriter(csvfile, output_names, output_types)

                writer.writeheader()

                i = 1
                keep_running = True
                trigger = False

                while keep_running:
                    if samples > 0 and i >= samples:
                        keep_running = False
                        logger.info("Recording complete")

                    if not use_trigger or trigger:
                        if buffered:
                            state = con.receive_buffered(binary)
                        else:
                            state = con.receive(binary)

                        if state is not None:
                            writer.writerow(state)
                            i += 1
                    else:
                        if buffered:
                            state = con.receive_buffered(False)
                        else:
                            state = con.receive(False)

                        if state is not None:
                            trigger = should_trigger(state)
                            if trigger:
                                logger.info("Recording triggered")

                    if state is None:
                        time.sleep(0.0005)

                    if not con.is_connected():
                        keep_running = False
                        abort_state = True
                        logger.info("Connection lost")

    except KeyboardInterrupt:
        abort_state = True
    except rtde.RTDEException:
        abort_state = True

    con.disconnect()
    return not abort_state
