import threading
import subprocess
import json
import logging
import os
import time

logger = logging.getLogger('daemon.' + __name__)

class RemoteAccessHandler:

    def __init__(self, device_client_instance):
        self.device_client = device_client_instance
        self.tunnel = None
        self.host="myurserver"
        self.home = "/root"
        self.ssh_config_file = os.path.join(self.home, ".ssh/ur_config")
        self.ssh_identity_file = os.path.join(self.home, ".ssh/ur_id_ed25519")
        self.ssh_public_key = self.ssh_identity_file + '.pub'
        self.ssh_known_hosts_file = os.path.join(self.home, ".ssh/known_hosts")
        self.tunnel_creation_status = "None"

    def create_ssh_tunnel(self, correlation_id, domain):
        logger.debug("Creating ssh tunnel for domain : " + str(domain))
        self.tunnel = threading.Thread(target=self._create_ssh_tunnel, args=(correlation_id, domain))
        self.tunnel.daemon = False
        self.tunnel.start()
        while self.tunnel_creation_status == "InProgress" :
            time.sleep(1)

        if self.tunnel_creation_status == "Success":
            self.tunnel_creation_status = "None"
            logger.debug("ssh tunnel created successfully")
            return True
        else:
            self.tunnel_creation_status = "None"
            logger.error("Failed to create ssh tunnel")
            return False

    def _create_ssh_tunnel(self, correlation_id, domain):
        self.tunnel_creation_status = "InProgress"
        try :
            host_name = domain.split("@")[1]
            conf = dict (
                HostName = str(host_name),
                User = str(domain.split("@")[0]),
                IdentityFile = self.ssh_identity_file,
                ServerAliveInterval = "60",
                ServerAliveCountMax = "3",
                ExitOnForwardFailure = "yes",
                ControlMaster = "auto",
                ControlPath = "/tmp/ssh-%r@%h:%p",
                ControlPersist = "no",
                StrictHostKeyChecking = "no",
                ConnectTimeout=10
            )

            # Write ssh config file
            with open(self.ssh_config_file, "w") as f:
                f.write("Host {0}\n".format(self.host))
                for key, value in conf.items():
                    f.write("\t%s %s\n".encode("utf-8") % (key, value))

            cmd_array = ["ssh", "-p 443", "-F", self.ssh_config_file, "-fNTCR", "0:localhost:6080", self.host]
            logger.debug("Creating ssh tunnel with command : " + str(cmd_array))
            subprocess.check_call(cmd_array)
            self.tunnel_creation_status = "Success"
            message = json.dumps({
                "type": "VncAccess",
                "messageType": "VncTunnelCreated",
                "correlationId": str(correlation_id)
            })
            self._send_message_to_hub(message)
        except Exception as e:
            self.tunnel_creation_status = "Failed"
            logger.exception("Error in creating ssh tunnel : " + str(e))
            message = json.dumps({
                "type": "VncAccess",
                "messageType": "VncTunnelCreationFailed",
                "correlationId": str(correlation_id)
            })
            self._send_message_to_hub(message)
    
    def send_remote_request_access_response(self, correlation_id, accepted):
        logger.debug("send_remote_request_access_response for request id : " + str(correlation_id))
        if accepted == True:
            try:

                # Generate a new key pair
                cmd = "bash -c \"ssh-keygen -t ed25519 -f /root/.ssh/ur_id_ed25519 -q -N '' <<< y\""
                subprocess.check_output(cmd, shell=True, stderr=subprocess.STDOUT)
                logger.debug("Created new ssh key: %s", self.ssh_identity_file)

                # Read public key
                with open(self.ssh_public_key, "r") as f:
                    publicKey = f.read()

                # Send accepted response message to iot hub
                message = json.dumps({
                    "type": "VncAccess",
                    "messageType": "VncRequestUserResponse",
                    "AccessRequestAccepted": "True",
                    "correlationId": str(correlation_id),
                    "PublicKey": str(publicKey)
                })
                self._send_message_to_hub(message)
                return True
            
            except subprocess.CalledProcessError as calledProcessError:
                logger.exception("Failed to send remote access request response:")
                logger.error(calledProcessError.output)
                raise e

            except Exception as e:
                logger.exception("Failed to send remote access request response:")
                raise e

        # If the access request is not accepted
        message = json.dumps({
            "type": "VncAccess",
            "messageType": "VncRequestUserResponse",
            "AccessRequestAccepted": "False",
            "correlationId": str(correlation_id)
        })
        self._send_message_to_hub(message)


    def send_remote_access_client_connected_event(self, correlation_id):
        message = json.dumps({
            "type": "VncAccess",
            "messageType": "VncClientConnected",
            "correlationId": str(correlation_id)
        })
        self._send_message_to_hub(message)

    def close_ssh_tunnel(self, correlation_id):
        logger.debug("Closing ssh tunnel for request id : " + str(correlation_id))
        message = json.dumps({
            "type": "VncAccess",
            "messageType": "VncTunnelClosed",
            "correlationId": str(correlation_id)
        })
        self._send_message_to_hub(message)
        try:
            cmd_array = ["ssh", "-p 443", "-F", self.ssh_config_file, "-O", "exit", self.host]
            subprocess.check_call(cmd_array)
        except Exception as e:
            logger.exception("Failed to close ssh tunnel")
        self._delete_files()
        logger.debug("Closed ssh tunnel for request id : " + str(correlation_id))

    def _delete_files(self):
        try:
            if os.path.exists(self.ssh_identity_file):
                os.remove(self.ssh_identity_file)
            if os.path.exists(self.ssh_public_key):
                os.remove(self.ssh_public_key)
            if os.path.exists(self.ssh_config_file):
                os.remove(self.ssh_config_file)
        except Exception:
            logger.exception("Error in deleting files:")

    def _send_message_to_hub(self, message):
        try:
            self.device_client.send_message(message)
            return True
        except Exception as e:
            logger.error("Failed to send message : " + message + " to hub :" + str(e))
            return False

