import telnetlib
import gzip
import os
import json
import xml.etree.ElementTree as ET
import uuid
import shutil
import requests
import datetime
import logging

class LoadedProgramHandler:
    def __init__(self, upload_to_blob_handler):
        global logger
        logger = logging.getLogger('daemon.' + __name__)
        self.upload_to_blob_handler = upload_to_blob_handler

    def get_loaded_program_info(self):

        path_str = self.get_loaded_program_from_telnet()

        start_index = path_str.find('/')
        if start_index == -1:
            raise Exception("Program cannot be read from an unnamed program. Please save the program on the robot and try again")

        end_index = path_str.rindex('.')
        substr = path_str[start_index:end_index]
        orig_filepath = path_str[start_index:]
        start_index_filename = path_str.rindex('/') + 1
        filename = path_str[start_index_filename:]


        if "<" in filename:
            raise Exception("Program cannot be read from an unnamed program. Please save the program on the robot and try again")

        with gzip.open(orig_filepath, "rb") as gzip_file:
            xml_content = gzip_file.read()

        root = ET.fromstring(xml_content)
        root_attributes = root.attrib

        path = substr + ".script"
        size = os.path.getsize(path)

        program_info = {
            "programName": filename,
            "installation": root_attributes["installation"],
            "installationRelativePath": root_attributes["installationRelativePath"],
            "directory": root_attributes["directory"],
            "scriptSize": size
        }

        return json.dumps(program_info);

    def backup_program_file(self, abs_file_path, blob_folder, tags):

        if os.path.exists(abs_file_path) is False:
            raise Exception("File Not Found")

        if blob_folder == "":
            blob_folder = os.path.dirname(abs_file_path)
        if blob_folder[0] == "/":
            blob_folder = blob_folder[1:]
        self.upload_to_blob_handler.upload_files(blob_folder, abs_file_path, tags, False)

    def restore_program_file(self, abs_file_path, download_link):

        #target_folder = os.path.dirname(abs_file_path)
        #file_name = os.path.split(abs_file_path)[1]
        file_dir, file_name = os.path.split(abs_file_path)
        tmp_abs_file = os.path.join("/programs/.tmp", file_name)
        tmp_abs_path = os.path.normpath("/programs/.tmp")
        try:

            logger.debug ("path to file:" + abs_file_path)
            logger.debug ("Download link:" + download_link)

            os.makedirs( tmp_abs_path)

            with open(tmp_abs_file, "wb") as f:

                response = requests.get(download_link, stream=True, timeout=30)
                total_length = response.headers.get('content-length')

                if total_length is None: # no content length header
                    f.write(response.content)
                else:
                    dl = 0
                    for data in response.iter_content(chunk_size=4096):
                        dl += len(data)
                        f.write(data)

            if not os.path.exists(file_dir):
                os.makedirs(file_dir)
            os.close(os.open("/programs/.restore_"+file_name, os.O_CREAT))
            shutil.copy(tmp_abs_file, abs_file_path)
            shutil.rmtree(tmp_abs_path)

        except Exception as e:
            logger.error ("Error occured" + str(e))
            shutil.rmtree(tmp_abs_path)

    def get_program_hierarchy(self, ext):

        # traverse root directory, and list directories as dirs and files as files
        programs_path = self.get_programs_path()
        dic = {}
        for root, dirs, files in os.walk(programs_path):
            path = root.split(os.sep)
            for file in files:
                if os.path.splitext(file)[-1].lower() == ext:
                    current_directory = "/".join(path)
                    try:
                        d = dic[current_directory]
                    except:
                        dic[current_directory] = []
                        d = dic[current_directory]

                    d.append(str(file))

        programs_location = {
            "files": dic
        }

        return json.dumps(programs_location);

    def get_programs_path(self):
        return "/programs"

    def upload_loaded_program_script(self, tags):

        path_str = self.get_loaded_program_from_telnet()

        start_index = path_str.find('/')
        if start_index == -1:
            raise Exception("Program cannot be read from an unnamed program. Please save the program on the robot and try again")

        end_index = path_str.rindex('.')
        substr = path_str[start_index:end_index]
        start_index_filename = path_str.rindex('/') + 1
        script_path = substr + ".script"
        filename = path_str[start_index_filename:]

        if "<" in filename:
            raise Exception("Program cannot be read from an unnamed program. Please save the program on the robot and try again")

        guid = str(uuid.uuid4())
        directory = path_str[start_index:start_index_filename]
        new_path = directory + guid + ".script"
        script_file_name = path_str[start_index_filename:path_str.rindex(".")] + ".script"

        shutil.copyfile(script_path, new_path)

        self.upload_to_blob_handler.upload_files("ProgramScriptFiles", new_path, tags, False)

        os.remove(new_path)

        program_info = {
            "programName": script_file_name,
            "correlationId": guid
        }

        return json.dumps(program_info);

    def get_loaded_program_from_telnet(self):
        #Telnet stuff
        tn = telnetlib.Telnet("localhost", "29999")
        tn.write("get loaded program\n")
        tn.write("quit\n")
        d = tn.read_all()

        arr = d.split('\n')
        return arr[1];

    def _write_to_log_file(self, log_message):

        if self.is_robot():
            log_file_path = "/data/myUR/program_handler.log"
        else:
            log_file_path = "/tmp/program_handler.log"

        now = datetime.datetime.now()
        with open(log_file_path, 'a+') as f:
            f.write(str(now) + ": " + log_message + "\n")

    def is_robot(self):
        return os.path.exists("/root/ur-serial")