import time
from datetime import datetime
import subprocess
import os

def open_log(status, logFileName, directory):
    if not os.path.exists(directory):
        os.makedirs(directory)

    log = open(directory + logFileName, 'a')
    log.write(datetime.now().strftime("%d/%m/%Y %H:%M:%S") + ": " + status + "\n")
    return log

my_env = os.environ.copy()
my_env["DEBIAN_FRONTEND"] = "noninteractive"

def get_current_time():
    return datetime.now().strftime("%d/%m/%Y %H:%M:%S")

def log_to_file(logfile, message):
    logfile.write(get_current_time() + " : " + message + "\n")

def run_command(cmd, logfile, skipInstall, blocking=True):
    log_to_file(logfile, cmd)
    if not skipInstall:
        process = subprocess.Popen(cmd.split(), stdout=subprocess.PIPE, stderr=subprocess.STDOUT, env = my_env)
        def check_io():
            while True:
                output = process.stdout.readline().decode('utf-8').strip()
                if output:
                    log_to_file(logfile, output)
                else:
                    break

        while blocking and process.poll() is None:
            check_io()
        return process.returncode
    else:
        print(cmd + "\n")
        return 0

def build_package_list(path):
    from os import listdir
    from os.path import isfile, join
    files = [f for f in listdir(path) if isfile(join(path, f)) and isDebianFile(f)]
    cmd = ""
    for idx in range(len(files)):
        cmd = cmd + " " + path + files[idx]
    return cmd

def isDebianFile(f):
    from os.path import splitext
    root_ext = splitext(f)
    if root_ext[1] == ".deb" :
        return True
    return False

class JobManager:
    def __init__(self):
        self.status = "IDLE"
        self.statusFileName = "JobStatus.txt"
        self.progress = 0.
        self.steps_count = 0.
        self.steps_total = 0.
        self.skipInstall = False
        self.result = 1
        self.steps = []
        self.steps_blocking = []
        self.directory = ""

    def update_progress(self):
        if self.steps_total == 0.:
            self.progress = 0.
        else:
            self.progress = min(self.steps_count / self.steps_total, 1.)
        statusfile = open(self.directory + self.statusFileName, "w")
        statusfile.write(self.status + "\n")
        statusfile.write(str(self.progress))
        statusfile.close()


    def new_task(self, name, logFileName, skipInstall, directory, statusFile):
        self.statusFileName = statusFile
        self.progress = 0.
        self.steps_count = 0.
        self.steps_total = 0.
        self.status = name
        self.steps = []
        self.steps_blocking = []
        self.directory = directory
        self.log = open_log(name, logFileName, self.directory)
        self.skipInstall = skipInstall
        self.update_progress()
        return self.log

    def add_step(self, cmd, blocking=True):
        self.steps.append(cmd)
        self.steps_blocking.append(blocking)
        self.steps_total = self.steps_total + 1.

    def inc_progress(self):
        self.steps_count = self.steps_count + 1.
        self.update_progress()
        if self.skipInstall:
            time.sleep(1)

    def execute(self):
        for idx in range(len(self.steps)):
            cmd = self.steps[idx]
            blocking = self.steps_blocking[idx]
            if isinstance(cmd, str):
                run_command(cmd, self.log, self.skipInstall, blocking)
            elif callable(cmd):
                cmd(self.log, self.skipInstall)
            self.inc_progress()

    def task_done(self):
        self.status = "IDLE"
        self.steps_count = self.steps_total
        self.update_progress()
        self.log.close()
        os.remove(self.directory + self.statusFileName)

