from azure.iot.device import MethodResponse
from generateFileHandler import GenerateFileHandler
from supportFileGenerator import SupportFileGenerator
from urServiceFileGenerator import UrServiceFileGenerator
import logging
from api_request import ApiRequest
from loadedProgramHandler import LoadedProgramHandler
from remoteUpdateHandler import RemoteUpdateHandler
import ast
import json


class IotHubMessageHandler:

    def __init__(self, device_client, upload_to_blob_handler_obj, remote_update_handler, wear_analysis_recorder_service):
        global logger
        logger = logging.getLogger('daemon.' + __name__)
        self.device_client = device_client
        self.upload_to_blob_handler = upload_to_blob_handler_obj
        self.support_file_handler = GenerateFileHandler(self, SupportFileGenerator(),
                                                        "support-file-process-state.json",
                                                        "SupportLogFiles",
                                                        "SupportFileStatus")
        self.ur_service_file_handler = GenerateFileHandler(self, UrServiceFileGenerator(),
                                                           "service-file-process-state.json",
                                                           "UrServiceFiles",
                                                           "UrServiceFileStatus")
        self.api_request = ApiRequest()
        self.loaded_program_handler = LoadedProgramHandler(self.upload_to_blob_handler)
        self.remote_update_handler = remote_update_handler
        self.wear_analysis_recorder_service = wear_analysis_recorder_service

    def receive_message_from_hub(self, message):

        try:
            if message.name == "generate_support_file":

                dict_args = ast.literal_eval(str(message.payload))
                tags = dict_args["tags"]
                support_file_handler_response = self.support_file_handler.generate_file(tags)
                payload = {"result": True, "correlationId": support_file_handler_response.correlationId }
                status = 200

            elif message.name == "upload":
                dict_args = ast.literal_eval(str(message.payload))
                abs_path = str(dict_args["abs_path"])
                blob_folder = str(dict_args["blob_folder"])
                tags = dict_args["tags"]
                self.upload_to_blob_handler.upload_files(blob_folder, abs_path, tags, True)
                status = 200
                payload = {"result": True, "data": "Uploaded requested file"}

            elif message.name == "backup-file-from-programs":
                dict_args = ast.literal_eval(str(message.payload))
                abs_path = str(dict_args["robot_path_to_file"])
                blob_folder = str(dict_args["blob_folder"])
                tags = dict_args["tags"]
                self.loaded_program_handler.backup_program_file(abs_path, blob_folder, tags)
                status = 200
                payload = {"result": True, "data": "Uploaded requested file"}

            elif message.name == "restore-program-file-from-backup":
                dict_args = ast.literal_eval(str(message.payload))
                abs_path = str(dict_args["robot_path_to_file"])
                download_link = str(dict_args["download_link"])
                self.loaded_program_handler.restore_program_file(abs_path, download_link)
                status = 200
                payload = {"result": True, "data": "Restore succeeded"}

            elif message.name == "get-program-variables":

                # get variables
                program_variables = self.api_request.get_program_variables()
                status = 200  # set return status code
                payload = {"result": True, "data": str(program_variables) }

            elif message.name == "get-loaded-program-info":

                # get program info
                program_info = self.loaded_program_handler.get_loaded_program_info()
                status = 200  # set return status code
                payload = {"result": True, "data": program_info }

            elif message.name == "get-program-file-hierarchy":

                dict_args = ast.literal_eval(str(message.payload))
                ext = str(dict_args["file_extension"])
                # get program info
                program_info = self.loaded_program_handler.get_program_hierarchy(ext)
                status = 200  # set return status code
                payload = {"result": True, "data": program_info }

            elif message.name == "download-update-remotely":

                dict_args = ast.literal_eval(str(message.payload))
                file_name = str(dict_args["file_name"])
                file_url = str(dict_args["file_url"])
                file_size = str(dict_args["file_size"])
                target_version = str(dict_args["target_version"])
                override = str(dict_args["override"]).lower() == "true"

                remote_update_handler_response = self.remote_update_handler.start_download_update(file_url, file_name, file_size, target_version, override)

                payload = {"result": True, "correlationId": remote_update_handler_response.correlationId }
                status = 200  # set return status code


            elif message.name == "update-remotely":

                dict_args = ast.literal_eval(str(message.payload))
                target_version = str(dict_args["target_version"])
                logger.debug("target_version: " + target_version)

                self.remote_update_handler.validate_update_request_for_target_version(target_version)
                self.api_request.request_to_start_software_update(target_version)

                start_update_response = {
                    "updateStarted" : True,
                    "message" : "Request for update sent, waiting for approval on the teach pendant."
                }
                response = json.dumps(start_update_response)
                logger.debug("start_update_response: " + response)
                payload = {"result": True, "data":  str(response)}
                status = 200  # set return status code

            elif message.name == "request-vnc-access":
                dict_args = ast.literal_eval(str(message.payload))
                requesters_name = str(dict_args["requesters_name"])
                remote_access_response_correlation_id = self.api_request.request_remote_access(requesters_name)
                payload = {"result": True, "correlationId": str(remote_access_response_correlation_id) }
                status = 200  # set return status code

            elif message.name == "request-start-tunnel":
                dict_args = ast.literal_eval(str(message.payload))
                requesters_name = str(dict_args["requesters_name"])
                domain = str(dict_args["domain"])
                remote_access_response = self.api_request.request_start_remote_access_session(requesters_name, domain)
                payload = {"result": True, "data": str(remote_access_response), "correlationId": str(remote_access_response)}
                status = 200  # set return status code

            elif message.name == "end-vnc-session":
                end_remote_session_response = self.api_request.end_remote_session()
                payload = {"result": True, "data": str(end_remote_session_response) }
                status = 200  # set return status code

            elif message.name == "upload-loaded-program-script":
                # get program info
                dict_args = ast.literal_eval(str(message.payload))
                tags = dict_args["tags"]
                blob_info = self.loaded_program_handler.upload_loaded_program_script(tags)
                status = 200  # set return status code
                payload = {"result": True, "data": blob_info }

            elif message.name == "download": #For testing download, not used anywhere
                dict_args = ast.literal_eval(str(message.payload))
                abs_save_path = str(dict_args["abs_save_path"])
                file_url = str(dict_args["file_url"])
                if not file_url.startswith("https://fleetblobdev.blob.core.windows.net/updates"):
                    raise ValueError("The provided url is not recognized as a UR IoT hub domain.")
                r = requests.get(file_url, allow_redirects=False)
                with open(abs_save_path, 'wb') as f:
                    f.write(r.content)
                status = 200  # set return status code
                payload = {"result": True, "data": "Downloaded file Successful"}

            elif message.name == "notify-urcap-update-available":
                dict_args = ast.literal_eval(str(message.payload))
                urcap_version_new = str(dict_args["urcap_version_new"])
                urcap_version_current = str(dict_args["urcap_version_current"])

                urcap_update_response_correlation_id = self.api_request.notify_urcap_update_available(urcap_version_new, urcap_version_current)

                payload = {"result": True, "correlationId": str(urcap_update_response_correlation_id) }
                status = 200  # set return status code

            elif message.name == "generate_service_file":
                dict_args = ast.literal_eval(str(message.payload))
                tags = dict_args["tags"]
                ur_service_file_handler_response = self.ur_service_file_handler.generate_file(tags)
                payload = {"result": True, "correlationId": ur_service_file_handler_response.correlationId }
                status = 200

            elif message.name == "joint_monitoring_service":
                dict_args = ast.literal_eval(str(message.payload))
                enable_param = str(dict_args["enable"]).lower() == 'true'
                self.wear_analysis_recorder_service.enable_monitoring(enable_param)
                status = 200
                if enable_param:
                    payload = {"result": True, "data": "Enable joint monitoring succeeded"}
                else:
                    payload = {"result": True, "data": "Disable joint monitoring succeeded"}

            elif message.name == "joint_monitoring_service_recording_length":
                dict_args = ast.literal_eval(str(message.payload))
                length = dict_args.get("recording_length")
                self.wear_analysis_recorder_service.set_recording_length(length)
                payload = {"result": True, "data": "Recording length successfully updated" }
                status = 200

            elif message.name == "joint_monitoring_service_recording_schedule":
                dict_args = ast.literal_eval(str(message.payload))
                weeks = dict_args.get("weeks")
                days = dict_args.get("days")
                hours = dict_args.get("hours")
                minutes = dict_args.get("minutes")
                self.wear_analysis_recorder_service.set_recording_interval(weeks, days, hours, minutes)
                payload = {"result": True, "data": "Recording schedule successfully updated" }
                status = 200

            elif message.name == "joint_monitoring_service_status":
                st = self.wear_analysis_recorder_service.get_status()
                payload = {"result": True, "status": st }
                status = 200

            else:
                payload = {"result": False, "data": "Method not found. Please update your URCap."}
                status = 404
        except Exception as e:
            logger.exception('Unhandled error in direct message')
            payload = {"result": False, "data":  str(e)}
            status = 400
        finally:
            method_response = MethodResponse.create_from_method_request(message, status, payload)
            self.device_client.send_method_response(method_response)

    def send_message_to_hub(self, message):
        try:
            self.device_client.send_message(message)
            return True
        except KeyboardInterrupt:
            logger.error("User initiated exit")
            return False
        except Exception as e:
            logger.error("Failed to send message : " + message + " to hub :" + e)
            return False

    def subscribe_listening_for_messages(self):
        self.device_client.on_method_request_received = self.receive_message_from_hub
        return True

